# -*- coding: utf-8 -*-

"""
	Orion
    https://orionoid.com

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

"""

########################################################################################################################################
# FOR DEVELOPERS
########################################################################################################################################

To use the Orion Kodi addon, do the following:
	1. Add Orion as a dependency to your addon.xml:
			<import addon="script.module.orion" version="1.0.0" />
	2. Import the Orion module in your Python script:
			from orion import *
	3. Create a new Orion object with your app API key:
			orion = Orion('my_app_key')
	4. Search for the streams using the instance from the previous step:
			results = orion.streams(type = Orion.TypeMovie, idXYZ = 'Orion, IMDb, TMDb, or TVDb ID')

A few things to note:
	1. Do not name your file "orion.py" or your class "Orion", because this will clash with Orion's import.
	2. A query requires a "type" and either and ID (idOrion, idImdb, idTmdb, idTvdb) or the "query" parameter.
		In addition, if you search for a show, you have to provide the "numberSeason" and "numberEpisode" together with the ID.

########################################################################################################################################

STREAM EXAMPLE 1 - Retrieve a movie using an IMDb ID.

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeMovie, idImdb = '0063350')

########################################################################################################################################

STREAM EXAMPLE 2 - Retrieve an episode using a TVDb ID.

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeShow, idTvdb = '73739', numberSeason = 3, numberEpisode = 5)

########################################################################################################################################

STREAM EXAMPLE 3 - Retrieve a movie using a query string. Using a query is not advised, since the wrong results might be returned.

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeMovie, query = 'Night of the Living Dead 1968')

########################################################################################################################################

STREAM EXAMPLE 4 - Retrieve a movie no larger than 2GB and being either a direct hoster link or cached on Premiumize.

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeMovie, idImdb = '0063350', fileSize = [None, 2147483648], access = [Orion.AccessDirect, Orion.AccessPremiumize])

########################################################################################################################################

STREAM EXAMPLE 5 - Retrieve a movie that has a video quality between SD and HD1080, and a DD or DTS audio codec.

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeMovie, idImdb = '0063350', videoQuality = [Orion.QualitySd, Orion.QualityHd1080], audioCodec = [Orion.CodecDd, Orion.CodecDts])

########################################################################################################################################

STREAM EXAMPLE 6 - Retrieve a movie that has a popularity of at least 50% and sorted by file size in descending order.

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeMovie, idImdb = '0063350', popularityPercent = 0.5, sortValue = Orion.SortFileSize, sortOrder = Orion.OrderDescending)

########################################################################################################################################

STREAM EXAMPLE 7 - Retrieve a movie with a maximum of 100 links and a page offset of 2 (that is link number 101 - 200).

	from orion import *
	result = Orion('my_app_key').streams(type = Orion.TypeMovie, idImdb = '0063350', limitCount = 100, limitPage = 2)

########################################################################################################################################

LINK - Retrieve Orion's website URL.

	from orion import *
	result = Orion('my_app_key').link()

########################################################################################################################################

APP DETAILS - Retrieve your app details and status.

	from orion import *
	result = Orion('my_app_key').app()

########################################################################################################################################

APP VALID - Check if your app key and status is valid.

	from orion import *
	result = Orion('my_app_key').appValid()

########################################################################################################################################

APP DIALOG - Show a Kodi dialog with your app details.

	from orion import *
	result = Orion('my_app_key').appDialog()

########################################################################################################################################

USER DETAILS - Retrieve your user details and status.

	from orion import *
	result = Orion('my_app_key').user()

########################################################################################################################################

USER VALID - Check if the user key and status is valid.

	from orion import *
	result = Orion('my_app_key').userValid()

########################################################################################################################################

USER FREE - Check if the user has a free account.

	from orion import *
	result = Orion('my_app_key').userFree()

########################################################################################################################################

USER PREMIUM - Check if the user has a premium account.

	from orion import *
	result = Orion('my_app_key').userPremium()

########################################################################################################################################

USER DIALOG - Show a Kodi dialog with the user details.

	from orion import *
	result = Orion('my_app_key').userDialog()

########################################################################################################################################

SERVER STATS - Retrieve the Orion server stats.

	from orion import *
	result = Orion('my_app_key').serverStats()

########################################################################################################################################

SERVER DIALOG - Show a Kodi dialog with the Orion server stats.

	from orion import *
	result = Orion('my_app_key').serverDialog()

########################################################################################################################################

SERVER TEST - Test if the Orion server is up and running.

	from orion import *
	result = Orion('my_app_key').serverTest()

########################################################################################################################################

"""

import xbmc, xbmcgui

# Import the Orion module.
from orion import *

# Get the app API key.
# In your own addon, you will hardcode your app API key in your code instead.
xbmcgui.Dialog().ok('Orion Example', 'This is an example addon to demonstrate to developers how Orion works. Please enter your [B]app[/B] API key in the next dialog. The app key can be found on Orion\'s website under the developer section.')
key = xbmcgui.Dialog().input('App API Key')
xbmc.executebuiltin('ActivateWindow(busydialog)')

# Create a new Orion object with the app API key.
# The detailed Orion class can be found in lib/orion/__init__.py
orion = Orion(key)

# This check is not neccessary.
if orion.appValid():

	# Search for the movie "Night of the Living Dead (1968)" using the IMDb ID.
	# We limit the results to a single link in order to not waste the daily limit for testing purposes.
	# All the remaining parameters/filters are retrieved from the settings sepcified by the user.
	# The parameters of this function can be manually overwritten by you, but it is better to leave it as the settings.
	results = orion.streams(Orion.TypeMovie, idImdb = '0063350', limitCount = 1)

	xbmcgui.Dialog().ok('Orion Results', 'A total of ' + str(results['count']['total']) + ' streams were found on Orion. ' + str(results['count']['filtered']) + ' of those streams were returned.')

	# Iterate over the returned results.
	for stream in results['streams']:
		# Do something with the streams.
		pass

xbmc.executebuiltin('Dialog.Close(busydialog)')
