# -*- coding: utf-8 -*-

"""
	Orion
    https://orionoid.com

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

##############################################################################
# ORIONSETTINGS
##############################################################################
# Class for handling the Kodi addon settings.
##############################################################################

import xbmcaddon
from orion.modules.oriontools import *

OrionSettingsSilent = False

class OrionSettings:

	##############################################################################
	# CONSTANTS
	##############################################################################

	ParameterDefault = 'default'
	ParameterValue = 'value'
	ParameterVisible = 'visible'

	CategoryGeneral = 0
	CategoryAccount = 1
	CategoryFilters = 2

	NotificationsDisabled = 0
	NotificationsEssential = 1
	NotificationsAll = 2

	ScrapingExclusive = 0
	ScrapingSequential = 1
	ScrapingParallel = 2

	##############################################################################
	# LAUNCH
	##############################################################################

	@classmethod
	def launch(self, category = None, section = None):
		OrionTools.execute('Addon.OpenSettings(%s)' % OrionTools.addonId())
		if not category == None: OrionTools.execute('SetFocus(%i)' % (int(category) + 100))
		if not section == None: OrionTools.execute('SetFocus(%i)' % (int(section) + 200))

	##############################################################################
	# PATH
	##############################################################################

	@classmethod
	def pathAddon(self):
		return OrionTools.pathJoin(OrionTools.addonPath(), 'resources', 'settings.xml')

	@classmethod
	def pathProfile(self):
		return OrionTools.pathJoin(OrionTools.addonProfile(), 'settings.xml')

	##############################################################################
	# SILENT
	##############################################################################

	@classmethod
	def silent(self):
		from orion.modules.orionuser import OrionUser
		global OrionSettingsSilent
		return OrionSettingsSilent or not OrionUser.instance().enabled()

	@classmethod
	def silentSet(self, silent = True):
		global OrionSettingsSilent
		OrionSettingsSilent = silent

	@classmethod
	def silentAllow(self, type = None):
		if self.silent():
			return False
		notifications = self.getGeneralNotificationsApi()
		if notifications == self.NotificationsDisabled:
			return False
		elif notifications == self.NotificationsAll:
			return True
		from orion.modules.orionapi import OrionApi
		return type == None or not type in OrionApi.Nonessential

	##############################################################################
	# DATA
	##############################################################################

	@classmethod
	def data(self):
		data = None
		path = OrionTools.pathJoin(self.pathAddon())
		with open(path, 'r') as file: data = file.read()
		return data

	##############################################################################
	# SET
	##############################################################################

	@classmethod
	def set(self, id, value):
		if value is True or value is False:
			value = OrionTools.toBoolean(value, string = True)
		elif OrionTools.isStructure(value):
			value = OrionTools.base64To(OrionTools.jsonTo(value))
		else:
			value = str(value)
		OrionTools.addon().setSetting(id = id, value = value)

	##############################################################################
	# GET
	##############################################################################

	@classmethod
	def get(self, id, raw = False, obfuscate = False):
		if raw:
			return self.getRaw(id = id, obfuscate = obfuscate)
		else:
			data = OrionTools.addon().getSetting(id)
			if obfuscate: data = OrionTools.obfuscate(data)
			return data

	@classmethod
	def getRaw(self, id, parameter = ParameterDefault, data = None, obfuscate = False):
		try:
			if data == None: data = self.data()
			indexStart = data.find(id)
			if indexStart < 0: return None
			indexStart = data.find('"', indexStart)
			if indexStart < 0: return None
			indexEnd = data.find('/>', indexStart)
			if indexEnd < 0: return None
			data = data[indexStart : indexEnd]
			indexStart = data.find(parameter)
			if indexStart < 0: return None
			indexStart = data.find('"', indexStart) + 1
			indexEnd = data.find('"', indexStart)
			data = data[indexStart : indexEnd]
			if obfuscate: data = OrionTools.obfuscate(data)
			return data
		except:
			return None

	@classmethod
	def getString(self, id, raw = False, obfuscate = False):
		return self.get(id = id, raw = raw, obfuscate = obfuscate)

	@classmethod
	def getBoolean(self, id, raw = False, obfuscate = False):
		return OrionTools.toBoolean(self.get(id = id, raw = raw, obfuscate = obfuscate))

	@classmethod
	def getBool(self, id, raw = False, obfuscate = False):
		return self.getBoolean(id = id, raw = raw, obfuscate = obfuscate)

	@classmethod
	def getNumber(self, id, raw = False, obfuscate = False):
		return self.getDecimal(id = id, raw = raw, obfuscate = obfuscate)

	@classmethod
	def getDecimal(self, id, raw = False, obfuscate = False):
		value = self.get(id = id, raw = raw, obfuscate = obfuscate)
		try: return float(value)
		except: return 0

	@classmethod
	def getFloat(self, id, raw = False, obfuscate = False):
		return self.getDecimal(id = id, raw = raw, obfuscate = obfuscate)

	@classmethod
	def getInteger(self, id, raw = False, obfuscate = False):
		value = self.get(id = id, raw = raw, obfuscate = obfuscate)
		try: return int(value)
		except: return 0

	@classmethod
	def getInt(self, id, raw = False, obfuscate = False):
		return self.getInteger(id = id, raw = raw, obfuscate = obfuscate)

	@classmethod
	def getList(self, id, raw = False, obfuscate = False):
		value = self.get(id = id, raw = raw, obfuscate = obfuscate)
		try:
			result = OrionTools.jsonFrom(OrionTools.base64From(value))
			return [] if result == None else result
		except: return []

	@classmethod
	def getObject(self, id, raw = False, obfuscate = False):
		value = self.get(id = id, raw = raw, obfuscate = obfuscate)
		try: return OrionTools.jsonFrom(OrionTools.base64From(value))
		except: return None

	##############################################################################
	# GET CUSTOM
	##############################################################################

	@classmethod
	def getGeneralNotificationsApi(self):
		return self.getInteger('general.notifications.api')

	@classmethod
	def getGeneralNotificationsNews(self):
		return self.getBoolean('general.notifications.news')

	@classmethod
	def getGeneralScrapingTimeout(self):
		return self.getInteger('general.scraping.timeout')

	@classmethod
	def getGeneralScrapingMode(self):
		return self.getInteger('general.scraping.mode')

	@classmethod
	def getGeneralScrapingCount(self):
		return self.getInteger('general.scraping.count')

	@classmethod
	def getFilters(self, type, include = False, exclude = False):
		values = self.getObject(type)
		try:
			if include: values = [key for key, value in values.iteritems() if value['enabled']]
		except: pass
		try:
			if exclude: values = [key for key, value in values.iteritems() if not value['enabled']]
		except: pass
		return values if values else [] if (include or exclude) else {}

	@classmethod
	def getFiltersStreamSource(self, include = False, exclude = False):
		return self.getFilters('filters.stream.source', include = include, exclude = exclude)

	@classmethod
	def getFiltersMetaRelease(self, include = False, exclude = False):
		return self.getFilters('filters.meta.release', include = include, exclude = exclude)

	@classmethod
	def getFiltersMetaUploader(self, include = False, exclude = False):
		return self.getFilters('filters.meta.uploader', include = include, exclude = exclude)

	@classmethod
	def getFiltersMetaEdition(self, include = False, exclude = False):
		return self.getFilters('filters.meta.edition', include = include, exclude = exclude)

	@classmethod
	def getFiltersVideoCodec(self, include = False, exclude = False):
		return self.getFilters('filters.video.codec', include = include, exclude = exclude)

	@classmethod
	def getFiltersAudioType(self, include = False, exclude = False):
		return self.getFilters('filters.audio.type', include = include, exclude = exclude)

	@classmethod
	def getFiltersAudioCodec(self, include = False, exclude = False):
		return self.getFilters('filters.audio.codec', include = include, exclude = exclude)

	@classmethod
	def getFiltersAudioLanguages(self, include = False, exclude = False):
		return self.getFilters('filters.audio.languages', include = include, exclude = exclude)

	@classmethod
	def getFiltersSubtitleType(self, include = False, exclude = False):
		return self.getFilters('filters.subtitle.type', include = include, exclude = exclude)

	@classmethod
	def getFiltersSubtitleLanguages(self, include = False, exclude = False):
		return self.getFilters('filters.subtitle.languages', include = include, exclude = exclude)

	##############################################################################
	# SET CUSTOM
	##############################################################################

	@classmethod
	def setFilters(self, values):
		self.setFiltersStreamSource(values)
		self.setFiltersMetaRelease(values)
		self.setFiltersMetaUploader(values)
		self.setFiltersMetaEdition(values)
		self.setFiltersVideoCodec(values)
		self.setFiltersAudioType(values)
		self.setFiltersAudioCodec(values)

	@classmethod
	def _setFilters(self, values, setting, functionStreams, functionGet):
		if not values: return
		items = {}
		try:
			from orion.modules.orionstream import OrionStream
			for value in values:
				attribute = getattr(value, functionStreams)()
				if not attribute == None:
					items[attribute.lower()] = {'name' : attribute.upper(), 'enabled' : True}
			settings = getattr(self, functionGet)()
			if settings:
				for key, value in items.iteritems():
					if not key in settings:
						settings[key] = value
				items = settings
		except:
			items = values
		if items: count = len([1 for key, value in items.iteritems() if value['enabled']])
		else: count = 0
		self.set(setting, items)
		self.set(setting + '.label', str(count) + ' ' + OrionTools.translate(32096))

	@classmethod
	def _setFiltersLanguages(self, values, setting, functionStreams, functionGet):
		if not values: return
		if values: count = len([1 for key, value in values.iteritems() if value['enabled']])
		else: count = 0
		self.set(setting, values)
		self.set(setting + '.label', str(count) + ' ' + OrionTools.translate(32096))

	@classmethod
	def setFiltersLimitCount(self, value):
		self.set('filters.limit.count', value)

	@classmethod
	def setFiltersLimitRetry(self, value):
		self.set('filters.limit.retry', value)

	@classmethod
	def setFiltersStreamSource(self, values):
		if not values: return
		items = {}
		try:
			from orion.modules.orionstream import OrionStream
			for value in values:
				attribute = value.streamSource()
				if not attribute == None:
					items[attribute.lower()] = {'name' : attribute.upper(), 'type' : value.streamType(), 'enabled' : True}
			settings = self.getFiltersStreamSource()
			if settings:
				for key, value in items.iteritems():
					if not key in settings:
						settings[key] = value
				items = settings
		except:
			items = values
		if items: count = len([1 for key, value in items.iteritems() if value['enabled']])
		else: count = 0
		self.set('filters.stream.source', items)
		self.set('filters.stream.source.label', str(count) + ' ' + OrionTools.translate(32096))

	@classmethod
	def setFiltersMetaRelease(self, values):
		self._setFilters(values, 'filters.meta.release', 'metaRelease', 'getFiltersMetaRelease')

	@classmethod
	def setFiltersMetaUploader(self, values):
		self._setFilters(values, 'filters.meta.uploader', 'metaUploader', 'getFiltersMetaUploader')

	@classmethod
	def setFiltersMetaEdition(self, values):
		self._setFilters(values, 'filters.meta.edition', 'metaEdition', 'getFiltersMetaEdition')

	@classmethod
	def setFiltersVideoCodec(self, values):
		self._setFilters(values, 'filters.video.codec', 'videoCodec', 'getFiltersVideoCodec')

	@classmethod
	def setFiltersAudioType(self, values):
		self._setFilters(values, 'filters.audio.type', 'audioType', 'getFiltersAudioType')

	@classmethod
	def setFiltersAudioCodec(self, values):
		self._setFilters(values, 'filters.audio.codec', 'audioCodec', 'getFiltersAudioCodec')

	@classmethod
	def setFiltersAudioLanguages(self, values):
		self._setFiltersLanguages(values, 'filters.audio.languages', 'audioLanguages', 'getFiltersAudioLanguages')

	@classmethod
	def setFiltersSubtitleType(self, values):
		self._setFilters(values, 'filters.subtitle.type', 'subtitleType', 'getFiltersSubtitleType')

	@classmethod
	def setFiltersSubtitleLanguages(self, values):
		self._setFiltersLanguages(values, 'filters.subtitle.languages', 'subtitleLanguages', 'getFiltersSubtitleLanguages')
